package com.sprd.pacextractor;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;

public class PacExtractor {

    public static final short SIZE_OF_PAC_HEADER = 2124; // bytes
    public static final short SIZE_OF_PARTITION_HEADER = 2580; // bytes
    private static RandomAccessFile in;

    public static String getString(short[] baseString) {
        if (baseString == null || baseString[0] == 0)
            return "";
        byte[] sb = new byte[baseString.length];
        for (int i = 0; i < sb.length; ++i)
            sb[i] = new Integer(unsigned(baseString[i])).byteValue();
        return new String(sb).trim();
    }

    public static void main(String[] args) {
        if (args.length < 2) {
            System.err.println("Usage: PacExtractor [-d] [-c] firmware.pac outdir");
            System.err.println("\t -d\t enable debug output");
            System.err.println("\t -c\t compute and verify CRC16");
            System.exit(2);
        }
        boolean debug = false, checkCRC16 = false;
        if (args[0].equals("-d") || args[1].equals("-d")) debug = true;
        if (args[1].equals("-c") || args[0].equals("-c")) checkCRC16 = true;

        try {
            File inFile = new File(args[args.length - 2]);
            if (inFile.length() < SIZE_OF_PAC_HEADER) {
                System.err.println(inFile + " is not a PAC firmware.");
            	System.exit(1);
            }

            File outdir = new File(args[args.length - 1]);
            if (!outdir.exists()) outdir.mkdirs();
            if (outdir.isFile()) {
                System.err.println("File with name " + outdir + " exists.");
            	System.exit(1);
            }
            String outdirS = outdir.getAbsolutePath() + File.separator;

            in = new RandomAccessFile(inFile, "r");
            PacHeader pacHeader = new PacHeader();
            if (debug) pacHeader.printInfo();
            if (unsigned(pacHeader.dwSize) != inFile.length()) {
                System.err.println("Bin packet's size is not correct");
            	System.exit(1);
            }

            if (checkCRC16) CheckCRC16.checkCRC16(in, debug);

            in.seek(unsigned(pacHeader.partitionsListStart));
            PartitionHeader[] partHeaders = new PartitionHeader[pacHeader.partitionCount];
            int i;
            for (i = 0; i < partHeaders.length; i++) {
                partHeaders[i] = new PartitionHeader();
                if (debug) partHeaders[i].printInfo();
            }

            System.out.println("\nExtracting...\n");
            byte[] buffer = new byte[4096];
            for (i = 0; i < partHeaders.length; i++) {
                if (partHeaders[i].partitionSize == 0)
                    continue;
                in.seek(unsigned(partHeaders[i].partitionAddrInPac));
                String outfile = getString(partHeaders[i].fileName);
                System.out.println(outfile);

                FileOutputStream fos = new FileOutputStream(outdirS + outfile);
                int read;
                long len = unsigned(partHeaders[i].partitionSize);
                while ((read = in.read(buffer, 0, len < buffer.length ? (int) len : buffer.length)) > 0) {
                    fos.write(buffer, 0, read);
                    len -= read;
                }
                fos.close();
            }

            System.out.println("\nDone...");
        } catch (Exception e) {
            e.printStackTrace();
            System.exit(1);
        } finally {
            try {
                if (in != null) in.close();
            } catch (IOException e) {
                e.printStackTrace();
                System.exit(1);
            }
        }
    }

    public static int unsigned(short n) {
        return n & 0xffff;
    }

    public static long unsigned(int n) {
        return n & 0xffffffffL;
    }

    public static int readInt() throws IOException{
        int result = 0;
        for (int i = 0; i < 4; i++) {
            result |= (in.readUnsignedByte() << (8 * i));
        }
        return result;
    }

    public static short readShort() throws IOException {
        short result = 0;
        for (int i = 0; i < 2; i++) {
            result |= (in.readUnsignedByte() << (8 * i));
        }
        return result;
    }

    public static class PacHeader {
        short[] szVersion = new short[24]; // packet struct version
        int dwSize; // the whole packet size
        short[] productName = new short[256]; // product name
        short[] firmwareName = new short[256]; // product version
        int partitionCount; // the number of files that will be downloaded, the file may be an operation
        int partitionsListStart; // the offset from the packet file header to the array of PartitionHeader struct buffer
        int dwMode;
        int dwFlashType;
        int dwNandStrategy;
        int dwIsNvBackup;
        int dwNandPageType;
        short[] szPrdAlias = new short[100]; // product alias
        int dwOmaDmProductFlag;
        int dwIsOmaDM;
        int dwIsPreload;
        int[] dwReserved = new int[200];
        int dwMagic;
        short wCRC1;
        short wCRC2;

        public PacHeader() throws IOException {
            int i;
        	for (i = 0; i < szVersion.length; ++i)
        	    szVersion[i] = readShort();
        	dwSize = readInt();
        	for (i = 0; i < productName.length; ++i)
        	    productName[i] = readShort();
        	for (i = 0; i < firmwareName.length; ++i)
        	    firmwareName[i] = readShort();
        	partitionCount = readInt();
        	partitionsListStart = readInt();
        	dwMode = readInt();
        	dwFlashType = readInt();
        	dwNandStrategy = readInt();
        	dwIsNvBackup = readInt();
        	dwNandPageType = readInt();
        	for (i = 0; i < szPrdAlias.length; ++i)
        	    szPrdAlias[i] = readShort();
        	dwOmaDmProductFlag = readInt();
        	dwIsPreload = readInt();
        	dwIsOmaDM = readInt();
        	for (i = 0; i < dwReserved.length; ++i)
        	    dwReserved[i] = readInt();
            dwMagic = readInt();
        	wCRC1 = readShort();
        	wCRC2 = readShort();
        }

        public void printInfo() {
            System.out.println(
              "Version\t\t= " + getString(szVersion) +
              "\nSize\t\t= " + unsigned(dwSize) +
              "\nPrdName\t\t= " + getString(productName) +
              "\nFirmwareName\t= " + getString(firmwareName) +
              "\nFileCount\t= " + unsigned(partitionCount) +
              "\nFileOffset\t= " + unsigned(partitionsListStart) +
              "\nMode\t\t= " + dwMode +
              "\nFlashType\t= " + dwFlashType +
              "\nNandStrategy\t= " + dwNandStrategy +
              "\nIsNvBackup\t= " + dwIsNvBackup +
              "\nNandPageType\t= " + dwNandPageType +
              "\nPrdAlias\t= " + getString(szPrdAlias) +
              "\nOmaDmPrdFlag\t= " + dwOmaDmProductFlag +
              "\nIsOmaDM\t\t= " + dwIsOmaDM +
              "\nIsPreload\t= " + dwIsPreload +
              "\nMagic\t\t= 0x" + Long.toHexString(unsigned(dwMagic)) +
              "\nCRC1\t\t= " + unsigned(wCRC1) +
              "\nCRC2\t\t= " + unsigned(wCRC2) + "\n\n"
            );
        }
    }

    public static class PartitionHeader {
        int length; // size of this struct itself
        short[] partitionName = new short[256]; // file ID,such as FDL,Fdl2,NV and etc.
        short[] fileName = new short[256]; // file name in the packet bin file. It only stores file name
        short[] szFileName = new short[256]; // Reserved now
        int partitionSize; // file size
        int nFileFlag; // if "0", means that it need not a file, and
                       // it is only an operation or a list of operations, such as file ID is "FLASH"
                       // if "1", means that it need a file
        int nCheckFlag; // if "1", this file must be downloaded;
                        // if "0", this file can not be downloaded;										
        int partitionAddrInPac; // the offset from the packet file header to this file data
        int dwCanOmitFlag; // if "1", this file can not be downloaded and not check it as "All files"
                           // in download and spupgrade tool.
        int dwAddrNum;
        int[] dwAddr = new int[5];
        int[] dwReserved = new int[249]; // Reserved for future, not used now

        public PartitionHeader() throws IOException {
            length = readInt();
            int i;
            for (i = 0; i < partitionName.length; ++i)
                partitionName[i] = readShort();
            for (i = 0; i < fileName.length; ++i)
                fileName[i] = readShort();
            for (i = 0; i < szFileName.length; ++i)
                szFileName[i] = readShort();
            partitionSize = readInt();
            nFileFlag = readInt();
            nCheckFlag = readInt();
            partitionAddrInPac = readInt();
            dwCanOmitFlag = readInt();
            dwAddrNum = readInt();
            for (i = 0; i < dwAddr.length; ++i)
                dwAddr[i] = readInt();
            for (i = 0; i < dwReserved.length; ++i)
                dwReserved[i] = readInt();
        }

        public void printInfo() {
            System.out.println(
              "Size\t\t= " + unsigned(length) +
              "\nFileID\t\t= " + getString(partitionName) +
              "\nFileName\t= " + getString(fileName) +
              "\nFileSize\t= " + unsigned(partitionSize) +
              "\nFileFlag\t= " + nFileFlag +
              "\nCheckFlag\t= " + nCheckFlag +
              "\nDataOffset\t= " + unsigned(partitionAddrInPac) +
              "\nCanOmitFlag\t= " + dwCanOmitFlag + "\n"
            );
        }
    }
}
